<?php
/**
 * Badges hooks.
 *
 * @package Ecomus
 */

namespace Ecomus\WooCommerce;

use Ecomus\Helper, Ecomus\WooCommerce;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class of Badges
 */
class Badges {
	/**
	 * Instance
	 *
	 * @var $instance
	 */
	protected static $instance = null;

	/**
	 * Initiator
	 *
	 * @since 1.0.0
	 * @return object
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Instantiate the object.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function __construct() {
		add_filter( 'ecomus_woocommerce_badges_class', array( $this, 'woocommerce_badges_class' ), 10, 2 );
		remove_action( 'woocommerce_before_shop_loop_item_title', 'woocommerce_show_product_loop_sale_flash' );
		add_action( 'ecomus_product_before_loop_thumbnail', array( $this, 'badges' ), 2 );

		// Single product
		add_action( 'woocommerce_single_product_summary', array( $this, 'single_badges' ), 6 );
		add_action( 'ecomus_woocommerce_product_quickview_summary', array( $this, 'single_badges' ), 17 );

		// For product loop primary
		add_action( 'ecomus_product_loop_primary_thumbnail', array( $this, 'badges' ), 2 );

		// Add badges for data product
		add_filter( 'woocommerce_available_variation', array( $this, 'data_variation_badges' ), 10, 3 );
	}

	/**
	 * Product badges class
	 *
	 * @return void
	 */
	public function woocommerce_badges_class( $classes ) {
		if( Helper::get_option( 'badges_soldout_position' ) == 'center' ) {
			$classes .= ' sold-out--center';
		}

		if( in_array( \Ecomus\WooCommerce\Helper::get_product_card_layout(), ['3', '4'] ) ) {
			$classes .= ' woocommerce-badges--right';
		}

		return $classes;
	}

	/**
	 * Product badges.
	 */
	public static function badges( $product = null, $classes = '', $args = array() ) {
		if( empty( $product ) ) {
			global $product;
		}

		$badges = array();
		$custom_badges       = maybe_unserialize( get_post_meta( $product->get_id(), 'custom_badges_text', true ) );
		if ( $custom_badges ) {
			$style    = '';
			$custom_badges_bg    = get_post_meta( $product->get_id(), 'custom_badges_bg', true );
			$custom_badges_color = get_post_meta( $product->get_id(), 'custom_badges_color', true );
			$bg_color = ! empty( $custom_badges_bg ) ? '--id--badge-custom-bg:' . $custom_badges_bg . ';' : '';
			$color    = ! empty( $custom_badges_color ) ? '--id--badge-custom-color:' . $custom_badges_color . ';' : '';

			if ( $bg_color || $color ) {
				$style = 'style="' . $color . $bg_color . '"';
			}

			$badges['html']['custom'] = '<span class="custom woocommerce-badge"' . $style . '>' . esc_html( $custom_badges ) . '</span>';
		} else {
			$badges = self::get_badges( $product, $args );
		}

		if ( ! empty( $badges['html'] ) && ! empty( implode( '',$badges['html'] ) ) ) {
			$classes = ! empty( $badges['class'] ) ? $badges['class'] : $classes;
			printf( '<div class="woocommerce-badges %s">%s</div>', esc_attr( apply_filters( 'ecomus_woocommerce_badges_class', $classes ) ), implode( '', $badges['html'] ) );
		}

		$countdown = self::get_countdown( $args );
		echo ! empty( $countdown ) ? $countdown : '';
	}

	/**
	 * Single product badges.
	 */
	public static function single_badges( $product, $args = array(), $classes = 'woocommerce-badges--single woocommerce-badges--single-primary' ) {
		if( empty( $product ) ) {
			global $product;
		}
		$args = wp_parse_args(
			$args,
			array(
				'badges_sale'           => Helper::get_option( 'product_badges_sale' ),
				'badges_sale_type'      => Helper::get_option( 'product_badges_sale_type' ),
				'badges_new'            => Helper::get_option( 'product_badges_new' ),
				'badges_featured'       => Helper::get_option( 'product_badges_featured' ),
				'badges_in_stock'       => Helper::get_option( 'product_badges_stock' ),
				'badges_soldout'        => Helper::get_option( 'product_badges_stock' ),
				'badges_pre_order'      => Helper::get_option( 'product_badges_stock' ),
				'is_single'	            => true,
				'badges_sale_countdown' => false,
			)
		);

		self::badges( $product, $classes, $args );
	}

	/**
	 * Get product badges.
	 *
	 * @return array
	 */
	public static function get_badges( $product = array(), $args = array() ) {
		if( empty( $product ) ) {
			global $product;
		}

		$args = wp_parse_args(
			$args,
			array(
				'badges_soldout'        => Helper::get_option( 'badges_soldout' ),
				'badges_soldout_text'   => esc_html__( 'Sold out', 'ecomus' ),
				'badges_sale'           => Helper::get_option( 'badges_sale' ),
				'badges_sale_type'      => Helper::get_option( 'badges_sale_type' ),
				'badges_sale_text'      => esc_html__( 'Sale', 'ecomus' ),
				'badges_new'            => Helper::get_option( 'badges_new' ),
				'badges_new_text'       => esc_html__( 'New', 'ecomus' ),
				'badges_featured'       => Helper::get_option( 'badges_featured' ),
				'badges_featured_text'  => esc_html__( 'Hot', 'ecomus' ),
				'badges_pre_order'      => Helper::get_option( 'badges_pre_order' ),
				'badges_pre_order_text' => esc_html__( 'Pre-Order', 'ecomus' ),
			)
		);

		$badges = array();
		$badges['class'] = $badges['countdown'] = '';

		if ( $args['badges_soldout'] && $product->get_stock_status() == 'outofstock' ) {
			$text               = ! empty( $args['badges_soldout_text'] ) ? $args['badges_soldout_text'] : esc_html__( 'Out Of Stock', 'ecomus' );
			$badges['html']['sold-out'] = '<div class="stock-badge"><p class="stock sold-out woocommerce-badge">' . esc_html( $text ) . '</p></div>';

			if( ! empty( $args['is_single'] ) ) {
				$badges['class']    = 'woocommerce-badges--single woocommerce-badges--single-primary sold-out';
			} else {
				$badges['class']    = 'sold-out';
			}

		} else {
			if( self::is_pre_order_active( $product ) && ($product->get_type() === 'simple' || $product->get_type() === 'variation') ) {
				$text          = empty( $text ) ? esc_html__( 'Pre-Order', 'ecomus' ) : $text;
				$badges['html']['pre_order'] = '<div class="stock-badge"><p class="stock pre-order  woocommerce-badge">' . esc_html( $text ) . '</p></div>';
				
			} else {
				if( $product->is_in_stock() && ! empty( $args['badges_in_stock'] ) && ! $product->is_on_backorder() && $product->is_purchasable() ) {
					$badges['html']['in_stock'] = '<div class="stock-badge"><p class="stock in-stock woocommerce-badge">' . wc_format_stock_for_display( $product ) . '</p></div>';
				}
			}


			if ( $product->is_on_sale() && $args['badges_sale'] ) {
				$badges['html']['sale'] = self::sale_flash( $product, $args );

				$classes = 'ecomus-badges-sale__countdown';
				$classes .= Helper::get_option( 'product_card_attribute_second' ) !== 'none' && ! empty( $product->get_attribute( esc_attr( Helper::get_option( 'product_card_attribute_second' ) ) ) ) ? ' ecomus-badges-sale__attribute-second' : '';

				if( ! isset( $args['badges_sale_countdown'] ) && Helper::get_option( 'badges_sale_countdown' ) ) {
					$badges['countdown'] = \Ecomus\WooCommerce\Helper::get_product_countdown( '', '', $classes, $product );
				}
			}

			else if ( $args['badges_new'] && in_array( $product->get_id(), WooCommerce\General::ecomus_woocommerce_get_new_product_ids() ) ) {
				$text          = $args['badges_new_text'];
				$text          = empty( $text ) ? esc_html__( 'New', 'ecomus' ) : $text;
				$badges['html']['new'] = '<span class="new woocommerce-badge">' . esc_html( $text ) . '</span>';
			}

			else if ( $product->is_featured() && $args['badges_featured'] ) {
				$text               = $args['badges_featured_text'];
				$text               = empty( $text ) ? esc_html__( 'Hot', 'ecomus' ) : $text;
				$badges['html']['featured'] = '<span class="featured woocommerce-badge">' . esc_html( $text ) . '</span>';
			}

			if( self::is_pre_order_active( $product ) && $product->get_type() === 'variable' ) {
				if( empty( $badges['html'] ) )				 {
					$text          = empty( $text ) ? esc_html__( 'Pre-Order', 'ecomus' ) : $text;
					$badges['html']['pre_order'] = '<div class="stock-badge"><p class="stock pre-order  woocommerce-badge">' . esc_html( $text ) . '</p></div>';
				}
			}

			if ( $args['badges_pre_order'] && $product->is_on_backorder() && empty( $badges['html']['pre_order'] ) ) {
				$text          = $args['badges_pre_order_text'];
				$text          = empty( $text ) ? esc_html__( 'Pre-Order', 'ecomus' ) : $text;
				$badges['html']['pre_order'] = '<div class="stock-badge"><p class="stock pre-order  woocommerce-badge">' . esc_html( $text ) . '</p></div>';
			}
		}


		$badges = apply_filters( 'ecomus_product_badges', $badges, $product );

		return $badges;
	}

	/**
	 * Sale badge.
	 *
	 * @param string $output  The sale flash HTML.
	 * @param object $post    The post object.
	 * @param object $product The product object.
	 *
	 * @return string
	 */
	public static function sale_flash( $product, $args ) {
		if ( 'grouped' == $product->get_type() ) {
			return '';
		}
		$output = '';
		$type       = $args[ 'badges_sale_type' ];
		$text       =  ! empty( $args['badges_sale_text'] ) ? $args['badges_sale_text'] : esc_html__( 'Sale', 'ecomus' );
		$percentage = 0;
		$saved      = 0;

		if ( 'percent' == $type || false !== strpos( $text, '{%}' ) || false !== strpos( $text, '{$}' ) ) {
			if ( $product->get_type() == 'variable' ) {
				$available_variations = $product->get_available_variations();
				$max_percentage       = 0;
				$max_saved            = 0;
				$total_variations     = count( $available_variations );

				for ( $i = 0; $i < $total_variations; $i++ ) {
					$variation_id        = $available_variations[ $i ]['variation_id'];
					$variable_product    = new \WC_Product_Variation( $variation_id );
					$regular_price       = $variable_product->get_regular_price();
					$sales_price         = $variable_product->get_sale_price();
					$variable_saved      = $regular_price && $sales_price ? ( $regular_price - $sales_price ) : 0;
					$variable_percentage = $regular_price && $sales_price ? round( ( ( ( $regular_price - $sales_price ) / $regular_price ) * 100 ) ) : 0;

					if ( $variable_saved > $max_saved ) {
						$max_saved = $variable_saved;
					}

					if ( $variable_percentage > $max_percentage ) {
						$max_percentage = $variable_percentage;
					}
				}

				$saved      = $max_saved ? $max_saved : $saved;
				$percentage = $max_percentage ? $max_percentage : $percentage;
			} else {
				$regular_price = wc_get_price_to_display($product, array( 'price' => $product->get_regular_price() ));
				$sale_price = wc_get_price_to_display($product);

				if( !empty( $regular_price ) && !empty( $sale_price ) && $regular_price > $sale_price ) {
					$saved      = $regular_price - $sale_price;
					$percentage = round( ( $saved / $regular_price ) * 100 );
				}
			}

		}

		if ( 'percent' == $type ) {
			if( $percentage >= 1 ) {
				$output = '<span class="onsale woocommerce-badge">-' . $percentage . '%</span>';
			}
		} else {
			$output = '<span class="onsale woocommerce-badge">' . wp_kses_post( $text ) . '</span>';
		}

		return $output;
	}

	/**
	 * Get sold of product deal
	 */
	public static function get_sold() {
		global $product;

		$limit = get_post_meta( $product->get_id(), '_deal_quantity', true );
		$sold  = intval( get_post_meta( $product->get_id(), '_deal_sales_counts', true ) );

		$output = ! empty( $limit ) ? '(' . esc_html( $sold ) . '/' . esc_html( $limit ) . ' ' . esc_html__( 'sold', 'ecomus' ) . ')' : '';

		return $output;
	}

	public static function get_countdown($args) {
		global $product;
		if ( isset( $args['badges_sale_countdown'] ) || ! Helper::get_option( 'badges_sale_countdown' ) ) {
			return;
		}
		if ( $product && $product->is_on_sale() ) {
			$classes = 'ecomus-badges-sale__countdown';
			$classes .= Helper::get_option( 'product_card_attribute_second' ) !== 'none' && ! empty( $product->get_attribute( esc_attr( Helper::get_option( 'product_card_attribute_second' ) ) ) ) ? ' ecomus-badges-sale__attribute-second' : '';

			return \Ecomus\WooCommerce\Helper::get_product_countdown( '', '', $classes, $product );
		}
	}

	public function data_variation_badges( $data, $parent, $variation ) {
		ob_start();
		$this->single_badges( $variation );
		$badges_html = ob_get_clean();
		$data['badges_html'] = esc_html($badges_html);
		return $data;
	}

	public static function is_pre_order_active($product = false) {
		if( ! class_exists( 'YITH_Pre_Order' ) ) {
			return false;
		}
		if( $product === false ) {
			global $product;
		}
		if( $product->is_type('simple') || $product->is_type('variation') ) {
			if( get_post_meta($product->get_id(),'_ywpo_preorder', true) === 'yes' ) {
				return true;
			}
		} elseif( $product->is_type('variable') ) {
			$variations = $product->get_children();
			foreach( $variations as $variation_id ) {
				if( get_post_meta($variation_id,'_ywpo_preorder', true) === 'yes' ) {
					return true;
				}
			}
		}
		return false;
	}
}
